/*
 * Created on Mar 16, 2005
 *
 * This file is for test purposes only.
 */

import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.math.BigInteger;
import java.security.MessageDigest;
import java.util.Properties;
import java.util.Random;
import java.util.StringTokenizer;

import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;

/**
 * @author Thomas J. Hansen, WirTek A/S
 *
 * This class can generate MEPD files based on property files. It can generate 3 different
 * formars:
 * 
 * 1) Format to be used by GenSML
 * 2) Format produced by GenSML
 * 3) Format produced by first-boot
 */
public class GenMEPD {
	public static void main(String[] args) throws Exception {
		if (args.length != 2) {
			System.err.println("arguments: <input file> <output file>");
			System.exit(1);
		}
		
		// Read property file
		FileInputStream  in  = new FileInputStream(args[0]);		
		Properties prop = new Properties();
		prop.load(in);
		in.close();

		// Get ready to write output
		FileOutputStream out = new FileOutputStream(args[1]);
		
		// Determine which type to make
		String type = getString(prop, "type");
		boolean genSML   = type.equals("GENSML");
		boolean external = type.equals("EXTERNAL");
		
		// Internal or external format
		if (genSML || external) {
			// Simulate GenSML?
			if (genSML) out.write(0x55);

			// Generate external lock structure
			ByteArrayOutputStream mepd = new ByteArrayOutputStream();
			int categories = writeMepdConf(mepd, prop);
			int seeds = 1; // Always generate a failure counter reset key seed
			for (int i = 0; i < categories; i++) {
				if (writeCategoryHeader(mepd, prop, i)) seeds++;
				writeCategoryData(mepd, prop, i);
			}
			byte data[] = mepd.toByteArray();

			// Use encryption?
			boolean desEncrypt = prop.getProperty("des.key") != null;
			boolean rsaEncrypt = prop.getProperty("rsa.modulus") != null;
			BigInteger cleartext = new BigInteger("0");

			if (desEncrypt) {
				// Generate DES key
				SecretKeySpec desKey = new SecretKeySpec(getString(prop, "des.key").getBytes(),"DES");
				cleartext = packBigInteger(desKey.getEncoded());

				// Encrypt MEPD data
				Cipher desCipher = Cipher.getInstance("DES/ECB/NoPadding");
				desCipher.init(Cipher.ENCRYPT_MODE, desKey);
				data = desCipher.doFinal(data);
			}

			// Calculate checksum
			MessageDigest md = MessageDigest.getInstance("SHA");
			md.update((byte)(data.length & 0xFF));
			md.update((byte)(data.length / 256));
			byte hash[] = md.digest(data);
			BigInteger num = packBigInteger(hash);
			cleartext = cleartext.or(num.shiftLeft(64));

			if (rsaEncrypt) {
				BigInteger exp = new BigInteger(getString(prop, "rsa.exponent"), 16);
				BigInteger mod = new BigInteger(getString(prop, "rsa.modulus"), 16);
				num = cleartext.modPow(exp, mod);
			}

			// Write external MEPD file
			writeUINT8(out, seeds);
			writeBigInteger(out, rsaEncrypt ? num : cleartext, 128);
			writeUINT16(out, data.length);
			out.write(data);

			// Write seeds
			if (genSML) {
				Random rand = new Random();
				while(seeds-- > 0) {
					int value = rand.nextInt();
					writeUINT32(out, value);
				}
			}
		} else {
			// Write internal MEPD file
			writeGlobalConf(out, 0xAA);
			writeKey(out, prop, "fc.key");
			int categories = writeMepdConf(out, prop);
			for (int i = 0; i < categories; i++) {
				writeCategoryKey(out, prop, i);
				writeCategoryHeader(out, prop, i);
				writeCategoryData(out, prop, i);
			}
		}
		
		out.close();
	}

	private static String getString(Properties prop, String key) {
		String res = prop.getProperty(key);
		if (res == null) throw new RuntimeException(key + " not found in property file,");
		return res;
	}

	private static int getInteger(Properties prop, String key) {
		return Integer.parseInt(getString(prop, key));
	}
	
	private static int getFlags(Properties prop, String key) {
		int res = 0;
		StringTokenizer tok = new StringTokenizer(getString(prop, key), ",");
		while(tok.hasMoreElements()) {
			String bit = tok.nextToken();
			res |= (1 << Integer.parseInt(bit));
		}
		return res;
	}

	private static byte[] getData(Properties prop, String key) {
		String hex = getString(prop, key);
		byte bytes[] = new byte[hex.length() / 2];
		for (int i = 0; i < bytes.length; i++) {
			String sub = hex.substring(i*2, i*2 + 2);
			bytes[i] = (byte)Integer.parseInt(sub, 16);
		}
		return bytes;
	}

	private static void writeGlobalConf(OutputStream out, int magic) throws IOException {
		writeUINT8(out, magic);
		writePadding(out, 7);
	}
	
	private static int writeMepdConf(OutputStream out, Properties prop) throws IOException {
		int categories = getInteger(prop, "conf.numCategories");
		
		writeUINT16(out, getFlags(prop, "conf.flags"));
		writeUINT16(out, getFlags(prop, "conf.addNewMSI"));
		writeUINT8(out, categories);
		writeUINT8(out, getInteger(prop, "conf.fc.max"));
		writeUINT8(out, getInteger(prop, "conf.fc.current"));
		writeUINT8(out, getInteger(prop, "conf.fc.keyLength"));
		writeUINT8(out, getInteger(prop, "conf.mncLength"));
		writeUINT8(out, getInteger(prop, "conf.gid1Length"));
		writeUINT8(out, getInteger(prop, "conf.gid2Length"));
		writeUINT8(out, getInteger(prop, "conf.typeApprovalSIM"));		
		writeUINT8(out, getInteger(prop, "conf.testCPHS"));
		writePadding(out, 3);
		
		return categories;
	}

	private static void writeCategoryKey(OutputStream out, Properties prop, int index) throws IOException {
		writeKey(out, prop, "cat" + index + ".key");
	}
	
	private static boolean writeCategoryHeader(OutputStream out, Properties prop, int index) throws IOException {
		int flags = getFlags(prop, "cat" + index + ".flags");
		writeUINT8(out, getInteger(prop, "cat" + index + ".status"));
		writeUINT8(out, flags);
		writeUINT16(out, getFlags(prop, "cat" + index + ".dependency"));
		writeUINT16(out, getInteger(prop, "cat" + index + ".length"));
		writeUINT8(out, getInteger(prop, "cat" + index + ".keyLength"));
		writePadding(out, 1);
		return ((flags &0x2) != 0);
	}
	
	private static void writeCategoryData(OutputStream out, Properties prop, int index) throws IOException {
		byte data[] = getData(prop, "cat" + index + ".data");
		if (data.length > 256)
			throw new IllegalArgumentException("Category data too long.");
		out.write(data);
		writePadding(out, 256 - data.length);
	}
	
	private static void writeKey(OutputStream out, Properties prop, String name) throws IOException {
		String key = getString(prop, name);
		switch (key.length()) {
		case 16 :
			out.write(key.getBytes());
			break;
		case 0 :
			writePadding(out, 16);
			break;
		default:
			throw new RuntimeException("Illegal key length: " + name + " : " + key.length());
		}
	}
	
	private static void writeUINT8(OutputStream out, int value) throws IOException {
		if (value < 0 || value > 0xFF)
			throw new IllegalArgumentException("Illegal UINT8 value: " + value);
		out.write(value);
	}

	private static void writeUINT16(OutputStream out, int value) throws IOException {
		if (value < 0 || value > 0xFFFF)
			throw new IllegalArgumentException("Illegal UINT16 value: " + value);
		out.write(value & 0xFF);
		out.write(value >>> 8);
	}
	
	private static void writeUINT32(OutputStream out, int value) throws IOException {
		out.write(value & 0xFF);
		out.write((value >>> 8) & 0xFF);
		out.write((value >>> 16) & 0xFF);
		out.write((value >>> 24) & 0xFF);
	}
	
	private static void writeBigInteger(OutputStream out, BigInteger num, int max) throws IOException {
		byte bytes[] = num.toByteArray();
		int size   = bytes.length;
		// Take into account that there may be an extra byte to represent positive 1024 bit number
		int offset = (size > max) ? size - max : 0;
		// Write out bytes, LSB first
		while (size > offset)
			out.write(bytes[--size]);
		// Pad to achieve max width
		if (bytes.length < max)
			writePadding(out, max - bytes.length);
	}

	private static void writePadding(OutputStream out, int length) throws IOException {
		while (length-- > 0) out.write(0);
	}
	
	private static BigInteger packBigInteger(byte bytes[]) {
		byte rev[] = new byte[bytes.length];
		int j = 0;
		for (int i = bytes.length -1 ; i >= 0; i--)
			rev[j++] = bytes[i];
		return new BigInteger(1, rev);
	}
}
